IF EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[hips].[HealthProviderIndividualUpdate]') AND type in (N'P', N'PC'))
BEGIN
	DROP PROCEDURE [hips].[HealthProviderIndividualUpdate]
END
GO
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO

-- =============================================

-- Create date: 16 October 2012
-- Description:   Updates a record into the HealthProviderIndividuals Table
-- Modified:    09/29/2015 Added HealthProviderIndividualHpii
-- 05/02/2016 Allow HPI-I status to be updated and still work when the provider has not been linked to a hospital yet.
-- 10/02/2016 Properly store HPI-I modified date and user
-- 12/02/2016 Allow health provider individuals with no given names to be saved.
-- =============================================
CREATE PROCEDURE [hips].[HealthProviderIndividualUpdate] 
(
@HealthProviderIndividualId INT,
	@HealthProviderOrganisationNetworkId INT,
	@IamProviderIdentifier VARCHAR(30) = NULL,
    @HpiI VARCHAR(16) = NULL,
    @HpiiLastValidated DATETIME = NULL,
	@HpiiStatusId INT = -1,
    @TitleId INT = NULL,
    @FamilyName VARCHAR(80),
    @GivenNames VARCHAR (100) = NULL,
    @SuffixId INT = NULL,
	@UserModified varchar(256),
	@DateModified DATETIME,
	@LocalProviderCode VARCHAR(256) = NULL,
	@DeactivatedDate DATETIME = NULL
)
AS
BEGIN
   SET NOCOUNT ON
 DECLARE @intErrorCode int

    SET @intErrorCode = 0

    IF NOT EXISTS (SELECT TOP 1 [DateModified]
            	FROM [hips].[HealthProviderIndividual]
				WHERE [HealthProviderIndividualId] = @HealthProviderIndividualId AND [DateModified] = @DateModified)
    BEGIN
		DECLARE @DateModifiedText NVARCHAR(30)
		SET @DateModifiedText = CAST(@DateModified AS NVARCHAR)
		RAISERROR (50001, -1, -1, N'HealthProviderIndividual', N'HealthProviderIndividualId', @HealthProviderIndividualId, N'DateModified', @DateModifiedText)
        SET @intErrorCode = @@ERROR
    END
 IF @intErrorCode = 0
    BEGIN
        UPDATE [hips].[HealthProviderIndividual]
           SET [IamProviderIdentifier] = @IamProviderIdentifier
			,[TitleId] = @TitleId
			,[FamilyName] = @FamilyName
			,[GivenNames] = @GivenNames
			,[SuffixId] = @SuffixId
            ,[DateModified] = GetDate()
            ,[UserModified] = @UserModified
            ,[DeactivatedDate] = @DeactivatedDate
         WHERE [HealthProviderIndividualId] = @HealthProviderIndividualId
         
         UPDATE [hips].[HealthProviderIndividualHpii]
         SET [HpiI] = @HpiI
            ,[HpiILastValidated] = @HpiiLastValidated
			,[HpiiStatusId] = @HpiiStatusId
			,[DateModified] = GetDate()
			,[UserModified] = @UserModified
         WHERE [HealthProviderIndividualId] = @HealthProviderIndividualId

        SELECT @intErrorCode = @@ERROR
    END

    IF @intErrorCode = 0
    BEGIN
		SELECT
			hcp.HealthProviderIndividualId,
			hcp.IamProviderIdentifier,
			hpih.HpiI,
			hpih.HpiILastValidated,
			hpih.HpiiStatusId,
			hcp.TitleId,
			t.Description AS Title,
			hcp.FamilyName,
			hcp.GivenNames,
			hcp.SuffixId,
			s.Description AS Suffix,
			hcp.DeactivatedDate,
			hcp.[DateCreated],
            hcp.[UserCreated],
            hcp.[DateModified],
            hcp.[UserModified]

		FROM hips.HealthProviderIndividual AS hcp 
		INNER JOIN hips.HealthProviderIndividualHpii AS hpih ON hcp.HealthProviderIndividualId = hpih.HealthProviderIndividualId AND hpih.HealthProviderOrganisationNetworkId = @HealthProviderOrganisationNetworkId 
		LEFT OUTER JOIN hips.Title AS t ON t.TitleId = hcp.TitleId
		LEFT OUTER JOIN hips.Suffix AS s ON s.SuffixId = hcp.SuffixId
		WHERE @HealthProviderIndividualId = hcp.HealthProviderIndividualId

    END

    RETURN(@intErrorCode)
END
GO


GO
GRANT EXEC ON [hips].[HealthProviderIndividualUpdate] TO RunStoredProcedure
GO
